<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

class UpcomingConferenceController extends Controller
{
    public function index(Request $request, \App\Services\ConferenceService $conferenceService)
    {
        $conferences = $conferenceService->getAllConferences();

        // Filter Logic
        // Process Data - Add dynamic fields if not already present in service
        $conferences = collect($conferences)->map(function ($conference) {
            // Recalculate if needed or just pass through. 
            // The service now provides formatted_date and deadline, but we can keep the calculation logic 
            // if we want to ensure consistency or if the service data changes format.
            // For now, let's trust the service or re-run the logic to be safe if the service raw data changes.

            if (!isset($conference['formatted_date']) || !isset($conference['deadline'])) {
                $cleanDate = str_replace(',', '', $conference['date']);
                $parts = explode(' ', $cleanDate);
                $month = $parts[0] ?? 'Jan';
                $dayRange = $parts[1] ?? '01';
                $year = $parts[2] ?? '2026';

                $day = intval($dayRange);
                $conference['formatted_date'] = str_pad($day, 2, '0', STR_PAD_LEFT) . ' ' . $month;

                $startDateString = "$day $month $year";
                $timestamp = strtotime($startDateString);
                $conference['deadline'] = $timestamp ? date('M d, Y', strtotime('-2 months', $timestamp)) : 'TBA';
            }

            return $conference;
        });

        $filteredConferences = $conferences;

        if ($request->has('search') && $request->search != '') {
            $filteredConferences = $filteredConferences->filter(function ($conference) use ($request) {
                return stripos($conference['title'], $request->search) !== false ||
                    stripos($conference['location'], $request->search) !== false;
            });
        }

        if ($request->has('year') && $request->year != '') {
            $filteredConferences = $filteredConferences->filter(function ($conference) use ($request) {
                return strpos($conference['date'], $request->year) !== false;
            });
        }

        if ($request->has('country') && $request->country != '') {
            $filteredConferences = $filteredConferences->filter(function ($conference) use ($request) {
                return $conference['country'] === $request->country;
            });
        }

        // Pagination Logic
        $perPage = 24;
        $page = $request->input('page', 1);
        $total = $filteredConferences->count();

        $paginatedConferences = new \Illuminate\Pagination\LengthAwarePaginator(
            $filteredConferences->forPage($page, $perPage)->values(),
            $total,
            $perPage,
            $page,
            ['path' => $request->url(), 'query' => $request->query()]
        );

        $years = collect($conferences)->map(function ($conference) {
            preg_match('/\d{4}/', $conference['date'], $matches);
            return $matches[0] ?? null;
        })->unique()->filter()->sort()->values();

        $countries = collect($conferences)->pluck('country')->unique()->sort()->values();

        return view('upcoming-conferences', [
            'conferences' => $paginatedConferences,
            'years' => $years,
            'countries' => $countries
        ]);
    }
}
